// SPDX-FileCopyrightText: Copyright (C) ARDUINO SRL (http://www.arduino.cc)
//
// SPDX-License-Identifier: MPL-2.0

#include <Arduino_RouterBridge.h>

// L298P电机驱动引脚定义
const int E1 = 5;  // 电机1使能/PWM
const int M1 = 4;  // 电机1方向
const int E2 = 6;  // 电机2使能/PWM
const int M2 = 7;  // 电机2方向


// 当前状态变量
int currentSpeed = 150;  // 默认速度（0-255）
int currentDir = 0;      // 0=停止, 1=前进, 2=后退, 3=左转, 4=右转

void setup() {

  // 初始化电机驱动引脚
  pinMode(M1, OUTPUT);
  pinMode(M2, OUTPUT);


  // 初始状态：停止
  stopMotors();

  // 初始化Bridge
  Bridge.begin();
  
  // 提供控制函数给Python端
  Bridge.provide("set_speed", set_speed);
  Bridge.provide("move_forward", move_forward);
  Bridge.provide("move_backward", move_backward);
  Bridge.provide("turn_left", turn_left);
  Bridge.provide("turn_right", turn_right);
  Bridge.provide("stop_motors", stop_motors);
  Bridge.provide("get_status", get_status);
}

void loop() {

  delay(10); // 小幅延时，减少CPU占用
}

// 设置速度
void set_speed(int speed) {
  if (speed >= 0 && speed <= 255) {
    currentSpeed = speed;
    // 如果当前在运动中，应用新速度
    if (currentDir != 0) {
      updateMotors();
    }
  }
}

// 前进
void move_forward() {
  currentDir = 1;
  digitalWrite(M1, HIGH);
  digitalWrite(M2, HIGH);
  analogWrite(E1, currentSpeed);
  if (currentSpeed+30 <= 255) {
  analogWrite(E2, currentSpeed+30);
  }
  else{
    analogWrite(E2, 255);
  }
 
}

// 后退
void move_backward() {
  currentDir = 2;
  digitalWrite(M1, LOW);
  digitalWrite(M2, LOW);
  analogWrite(E1, currentSpeed);
  if (currentSpeed+30 <= 255) {
  analogWrite(E2, currentSpeed+30);
  }
  else{
    analogWrite(E2, 255);
  }
}

// 左转
void turn_left() {
  currentDir = 3;
  digitalWrite(M1, HIGH);   // 右轮前进
  digitalWrite(M2, LOW);    // 左轮后退
  analogWrite(E1, currentSpeed);
  analogWrite(E2, currentSpeed);
}

// 右转
void turn_right() {
  currentDir = 4;
  digitalWrite(M1, LOW);    // 右轮后退
  digitalWrite(M2, HIGH);   // 左轮前进
  analogWrite(E1, currentSpeed);
  analogWrite(E2, currentSpeed);
}

// 停止
void stop_motors() {
  currentDir = 0;
  analogWrite(E1, 0);
  analogWrite(E2, 0);
}

// 更新电机（保持当前方向，应用新速度）
void updateMotors() {
  switch(currentDir) {
    case 1: move_forward(); break;
    case 2: move_backward(); break;
    case 3: turn_left(); break;
    case 4: turn_right(); break;
    default: stop_motors(); break;
  }
}

// 停止所有电机
void stopMotors() {
  analogWrite(E1, 0);
  analogWrite(E2, 0);
  
}

// 获取状态
int get_status() {
  // 返回组合状态：方向 * 1000 + 速度
  return currentDir * 1000 + currentSpeed;
}